/***********************************************************************/
/*                                                                     */
/* Name       : TCI_232.C           Version: 002.02023vpr              */
/*                                                                     */
/* Author     : Volker Pritsching                                      */
/*                                                                     */
/* Language   : C                                                      */
/*                                                                     */
/* Description: Communication interface for serial communication       */
/*              based on BCI_232.H from BIST_Software for mcb2         */
/*                                                                     */
/***********************************************************************/
/*                                                                     */
/* 16.09.98 V001 Gerhard Ort File created                              */
/* 18.07.04 V003 Volker Prisching RX ISR fr Fujitsu                   */
/*               Felherbehandlung gendert                             */
/***********************************************************************/
/***********************************************************************/
/* Imports                                                             */
/***********************************************************************/
#include <avr/io.h>
#include <avr/interrupt.h>
#include "systemcommandinterpreter_core.h"
/***********************************************************************/
/* Exports                                                             */
/***********************************************************************/
#include "System_RS232.h"
#define F_CPU 4000000UL	/* CPU clock in Hertz */
#define Baudrate 9600
/***********************************************************************/
/* Locals                                                              */
/***********************************************************************/
/*---------------------------------------------------------------------*/
/* Macros and defines                                                  */
/*---------------------------------------------------------------------*/
#define END_OF_SERIAL_STRING    0x0D    /* CR */
#define DEL                     0x7F
#define BACKSPACE               0x08
/*---------------------------------------------------------------------*/
/* Modulglobals                                                        */
/*---------------------------------------------------------------------*/
char t[COM_STRLEN_MAX+1];
char RX_Str_Ready;
static unsigned char ucCount = 0;
/***********************************************************************/
/* Functions                                                           */
/***********************************************************************/
/***********************************************************************/
/*                                                                     */
/* Function   : fSerial_putchar()                                      */
/*                                                                     */
/* Description: transmits a character using ASC0, the serial port      */
/*              '\n'-Character is changed to '\r' (for Terminal-Program*/
/*                                                                     */
/* Return     : character which should be transmitted                  */
/*                                                                     */
/***********************************************************************/
char fSerial_putchar
(
  char c            // IMP: character to send
){

  loop_until_bit_is_set(UCSRA, UDRE);
 // if(c=='\n') cI = '\r';
 // else           cI = c;
  UDR = c;
return c;
}
/***********************************************************************/
/*                                                                     */
/* Function   : fSerial_getkey()                                       */
/*                                                                     */
/* Description: receives a character using ASC0, the serial port       */
/*                                                                     */
/* Return     : received character                                     */
/*                                                                     */
/***********************************************************************/
char fSerial_getkey(void)
{
unsigned char c;

  c = UDR;
 // if (c > 0x30)  fSerial_putchar(c);

  if (bit_is_clear(UCSRA, FE)) return (c);
  return (-1);
}
/***********************************************************************/
/*                                                                     */
/* Function   : fSerialInit()                                          */
/*                                                                     */
/* Description: inits the RS232 communication - only local parameters  */
/*                                                                     */
/* Return     : 0 = no errror                                          */
/*              1 = communication error                                */
/*                                                                     */
/***********************************************************************/
unsigned short fSerialInit
(
  char  *pcParm  // IMP: parameter string (not used)
)                // base auto detect: 
{	/* initialize UART1 */
pcParm = 0;
  UCSRA = _BV(U2X);		/* improves baud rate error @ F_CPU = 1 MHz */
  UCSRB = _BV(TXEN)|_BV(RXEN)|_BV(RXCIE); /* tx/rx enable, rx complete intr */
  UBRRL = (F_CPU / (8 * 9600UL)) - 1;  /* 9600 Bd */
RX_Str_Ready = 0;
return 0;
}
/***********************************************************************/
/*                                                                     */
/* Function   : fSerialChange()                                        */
/*                                                                     */
/* Description: changes the communication parameter,                   */
/*                                                                     */
/* Return     : 0: no error                                            */
/*                                                                     */
/***********************************************************************/
unsigned short fSerialChange(void)
{
return NO_ERR;
}

/***********************************************************************/
/*                                                                     */
/* Function   : fSerialSendStr()                                       */
/*                                                                     */
/* Description: sends a string via serial interface (RS232)            */
/*              For this transmit direction no software handshake      */
/*                                                                     */
/* Return     : 0 = no errror                                          */
/*              COM_ERR = got NULL-Pointer or String too long,         */
/*              COM_STRLEN_MAX characters                              */
/*                                                                     */
/***********************************************************************/
unsigned short fSerialSendStr
(
  char *pc            // IMP: string to be send over the RS232
){
unsigned short ush;
ush = fSerialPureSend(pc);
fSerial_putchar(END_OF_SERIAL_STRING);
return ush;
}
/***********************************************************************/
/*                                                                     */
/* Function   : fSerialPureSend()                                      */
/*                                                                     */
/* Description: sends a string via serial interface (RS232)            */
/*              For this transmit direction no software handshake      */
/*                                                                     */
/* Return     : 0 = no errror                                          */
/*              COM_ERR = got NULL-Pointer or String too long,         */
/*              COM_STRLEN_MAX characters                              */
/*                                                                     */
/***********************************************************************/
unsigned short fSerialPureSend
(
  char *pc            // IMP: string to be send over the RS232
){
unsigned char uc;
if(pc==NULL) return COM_ERR;
for(uc=ucCOM_STRLEN_MAX; uc && *pc; uc--) fSerial_putchar(*pc++);
if(*pc) return COM_ERR;    // string was too long
else return NO_ERR;
}
/***********************************************************************/
/*                                                                     */
/* Function   : fSerialReceiveStr()                                    */
/*                                                                     */
/* Description: reads a command string from the RS232.                 */
/*   control until a string is received.                               */
/*  Each command string has to be terminated with END_OF_SERIAL_STRING */
/*  To make sure no data are lost the transmitter (=testsytem, terminal)*/
/*  there is a simple handshake. The transmitter has to wait for this echo*/
/*    before sending an new charakter.                                 */
/*                                                                     */
/* Return     : 0 = no errror                                          */
/*              1 = communication error                                */
/*                                                                     */
/***********************************************************************/

unsigned short fSerialReceiveStr
(
  char *pc      // EXP: string buffer  (max 80)
){
if (RX_Str_Ready){
  // copy input string to TSFW buffer, ucCount will be reset to 0
  ucCount--;  // the character END_OF_SERIAL_STRING is not part of input string
  pc[ucCount] = '\0';
  while(ucCount){
    ucCount--;
    pc[ucCount] = t[ucCount];
    RX_Str_Ready =0;
    }
  return NO_ERR;
  }
else return COM_NOT_FINISHED;   // there is no full command input
}

/***********************************************************************/

// __interrupt 
// void fSerialReceiveISR (void)

ISR(USART_RXC_vect)
{
unsigned char ucBuff;
ucBuff = fSerial_getkey();
if(ucCount > COM_STRLEN_MAX){
  ucCount = 0;    // reset for next input string
  }
if (RX_Str_Ready ==1){
    RX_Str_Ready = 0;
    ucCount =0;
    }
t[ucCount] = ucBuff;
//echo auch wenn gesendet wird Choas
if ( ucBuff == BACKSPACE || ucBuff == DEL ){  // process backspace and delete
  if( ucCount > 0 ){                          // last pos in t
    fSerial_putchar(BACKSPACE);
    fSerial_putchar(' ');
    fSerial_putchar(BACKSPACE);
    ucCount--;
    }
  }
else {                                        // write char into buffer
  fSerial_putchar(ucBuff);
  ucCount++;
  }
if(ucBuff == END_OF_SERIAL_STRING){
RX_Str_Ready = 1;
  }
}
